<?php
// Contact Form Handler for Velaxio Digital Solutions
// This file handles form submissions and sends emails

// Set content type to JSON
header('Content-Type: application/json');

// Enable CORS for cross-origin requests
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Configuration
$config = [
    'to_email' => 'hello@velaxio.com', // Change this to your email
    'from_email' => 'noreply@velaxio.com', // Change this to your domain email
    'subject_prefix' => '[Velaxio Contact] ',
    'max_message_length' => 2000,
    'required_fields' => ['name', 'email', 'service', 'message'],
    'allowed_services' => [
        'digital-marketing' => 'Digital Marketing',
        'ui-ux-design' => 'UI/UX Design',
        'web-hosting' => 'Web Hosting',
        'end-to-end-support' => 'End-to-End Support'
    ]
];

// Function to sanitize input
function sanitizeInput($input) {
    return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
}

// Function to validate email
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// Function to validate phone (optional field)
function validatePhone($phone) {
    if (empty($phone)) return true; // Phone is optional
    return preg_match('/^[\+]?[0-9\s\-\(\)]{10,}$/', $phone);
}

// Function to send email
function sendEmail($to, $subject, $message, $headers) {
    return mail($to, $subject, $message, $headers);
}

// Function to log contact form submissions
function logSubmission($data) {
    $logFile = 'contact_logs.txt';
    $logEntry = date('Y-m-d H:i:s') . ' - ' . json_encode($data) . "\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

// Initialize response
$response = ['success' => false, 'message' => ''];

try {
    // Get form data
    $formData = [
        'name' => isset($_POST['name']) ? sanitizeInput($_POST['name']) : '',
        'email' => isset($_POST['email']) ? sanitizeInput($_POST['email']) : '',
        'phone' => isset($_POST['phone']) ? sanitizeInput($_POST['phone']) : '',
        'service' => isset($_POST['service']) ? sanitizeInput($_POST['service']) : '',
        'message' => isset($_POST['message']) ? sanitizeInput($_POST['message']) : ''
    ];
    
    // Validate required fields
    $errors = [];
    
    foreach ($config['required_fields'] as $field) {
        if (empty($formData[$field])) {
            $errors[] = ucfirst($field) . ' is required';
        }
    }
    
    // Validate email
    if (!empty($formData['email']) && !validateEmail($formData['email'])) {
        $errors[] = 'Invalid email address';
    }
    
    // Validate phone
    if (!validatePhone($formData['phone'])) {
        $errors[] = 'Invalid phone number format';
    }
    
    // Validate service
    if (!empty($formData['service']) && !array_key_exists($formData['service'], $config['allowed_services'])) {
        $errors[] = 'Invalid service selection';
    }
    
    // Validate message length
    if (strlen($formData['message']) > $config['max_message_length']) {
        $errors[] = 'Message is too long (maximum ' . $config['max_message_length'] . ' characters)';
    }
    
    // Check for errors
    if (!empty($errors)) {
        $response['message'] = implode(', ', $errors);
        echo json_encode($response);
        exit();
    }
    
    // Prepare email content
    $serviceName = $config['allowed_services'][$formData['service']];
    
    $emailSubject = $config['subject_prefix'] . 'New Contact Form Submission';
    
    $emailMessage = "
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #00d4ff 0%, #4ecdc4 100%); color: white; padding: 20px; text-align: center; border-radius: 10px 10px 0 0; }
            .content { background: #f9f9f9; padding: 20px; border-radius: 0 0 10px 10px; }
            .field { margin-bottom: 15px; }
            .label { font-weight: bold; color: #00d4ff; }
            .value { margin-top: 5px; }
            .footer { text-align: center; margin-top: 20px; color: #666; font-size: 12px; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2>🚀 New Contact Form Submission</h2>
                <p>Velaxio Digital Solutions</p>
            </div>
            <div class='content'>
                <div class='field'>
                    <div class='label'>Name:</div>
                    <div class='value'>{$formData['name']}</div>
                </div>
                <div class='field'>
                    <div class='label'>Email:</div>
                    <div class='value'>{$formData['email']}</div>
                </div>
                <div class='field'>
                    <div class='label'>Phone:</div>
                    <div class='value'>" . (!empty($formData['phone']) ? $formData['phone'] : 'Not provided') . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Service Interest:</div>
                    <div class='value'>{$serviceName}</div>
                </div>
                <div class='field'>
                    <div class='label'>Message:</div>
                    <div class='value'>" . nl2br($formData['message']) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Submission Time:</div>
                    <div class='value'>" . date('Y-m-d H:i:s') . "</div>
                </div>
            </div>
            <div class='footer'>
                <p>This message was sent from the Velaxio Digital Solutions contact form.</p>
                <p>Please respond to the customer's email address: {$formData['email']}</p>
            </div>
        </div>
    </body>
    </html>
    ";
    
    // Prepare email headers
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        'From: ' . $config['from_email'],
        'Reply-To: ' . $formData['email'],
        'X-Mailer: PHP/' . phpversion(),
        'X-Priority: 3'
    ];
    
    // Send email
    $emailSent = sendEmail(
        $config['to_email'],
        $emailSubject,
        $emailMessage,
        implode("\r\n", $headers)
    );
    
    if ($emailSent) {
        // Log successful submission
        logSubmission($formData);
        
        // Send auto-reply to customer
        $autoReplySubject = 'Thank you for contacting Velaxio Digital Solutions';
        $autoReplyMessage = "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: linear-gradient(135deg, #00d4ff 0%, #4ecdc4 100%); color: white; padding: 20px; text-align: center; border-radius: 10px 10px 0 0; }
                .content { background: #f9f9f9; padding: 20px; border-radius: 0 0 10px 10px; }
                .footer { text-align: center; margin-top: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h2>🚀 Thank You for Contacting Us!</h2>
                    <p>Velaxio Digital Solutions</p>
                </div>
                <div class='content'>
                    <p>Dear {$formData['name']},</p>
                    <p>Thank you for your interest in our digital solutions services. We have received your message regarding <strong>{$serviceName}</strong> and will get back to you within 24 hours.</p>
                    <p>Our team of experts is ready to help you transform your business with:</p>
                    <ul>
                        <li>🤖 AI-Powered SEO (First in Industry)</li>
                        <li>🎨 Cutting-edge UI/UX Design</li>
                        <li>☁️ Reliable Web Hosting Solutions</li>
                        <li>🚀 End-to-End Digital Support</li>
                    </ul>
                    <p>In the meantime, feel free to explore our hosting plans and learn more about our innovative approach to digital marketing.</p>
                    <p>Best regards,<br>The Velaxio Team</p>
                </div>
                <div class='footer'>
                    <p>Velaxio Digital Solutions | hello@velaxio.com | +1 (555) 123-4567</p>
                </div>
            </div>
        </body>
        </html>
        ";
        
        $autoReplyHeaders = [
            'MIME-Version: 1.0',
            'Content-type: text/html; charset=UTF-8',
            'From: ' . $config['from_email'],
            'X-Mailer: PHP/' . phpversion()
        ];
        
        sendEmail(
            $formData['email'],
            $autoReplySubject,
            $autoReplyMessage,
            implode("\r\n", $autoReplyHeaders)
        );
        
        $response['success'] = true;
        $response['message'] = 'Message sent successfully! We\'ll get back to you within 24 hours.';
        
    } else {
        $response['message'] = 'Failed to send message. Please try again or contact us directly.';
    }
    
} catch (Exception $e) {
    // Log error
    error_log('Contact form error: ' . $e->getMessage());
    
    $response['message'] = 'An error occurred. Please try again later.';
}

// Return JSON response
echo json_encode($response);
?>
